<?php

namespace App\Http\Transformers;


use App\Helpers\Helper;
/**
 * Class PieChartTransformer
 *
 * This handles the standardized formatting of the API response we need to provide for
 * the pie charts
 *
 * @return \Illuminate\Http\Response
 * @since [v6.0.11]
 * @author [A. Gianotto] [<snipe@snipe.net>]
 */
class PieChartTransformer
{
    public function transformPieChartDate($totals)
    {

        $labels = [];
        $counts = [];
        $default_color_count = 0;
        $colors_array = [];

        foreach ($totals as $total) {

            if ($total['count'] > 0) {

                $labels[] = $total['label']." (".$total['count'].")";
                $counts[] = $total['count'];

                if (isset($total['color']) && !empty($total['color'])) {
                    $colors_array[] = $total['color'];
                } else {
                    // Use semantic colors for common status labels
                    $semantic_color = $this->getSemanticColorForStatus($total['label']);
                    if ($semantic_color) {
                        $colors_array[] = $semantic_color;
                    } else {
                        $colors_array[] = Helper::defaultChartColors($default_color_count);
                        $default_color_count++;
                    }
                }
            }
        }

        $results = [
            'labels' => $labels,
            'datasets' => [[
                'data' => $counts,
                'backgroundColor' => $colors_array,
                'hoverBackgroundColor' =>  $colors_array,
            ]],
        ];


        return $results;
    }

    /**
     * Get semantic colors for common status labels
     *
     * @param string $statusLabel
     * @return string|null
     */
    private function getSemanticColorForStatus($statusLabel)
    {
        $statusLabel = strtolower(trim($statusLabel));
        
        // Define semantic color mapping for common status labels
        $semanticColors = [
            // Ready/Deployable statuses - Green
            'ready to deploy' => '#28a745',
            'ready' => '#28a745',
            'deployable' => '#28a745',
            'available' => '#28a745',
            'active' => '#28a745',
            'in use' => '#28a745',
            'assigned' => '#28a745',
            
            // Pending/Warning statuses - Orange/Yellow
            'pending' => '#ffc107',
            'pending deployment' => '#ffc107',
            'waiting' => '#ffc107',
            'in progress' => '#ffc107',
            'maintenance' => '#ffc107',
            'repair' => '#ffc107',
            'under repair' => '#ffc107',
            
            // Error/Problem statuses - Red
            'broken' => '#dc3545',
            'damaged' => '#dc3545',
            'faulty' => '#dc3545',
            'error' => '#dc3545',
            'failed' => '#dc3545',
            'out of order' => '#dc3545',
            'not working' => '#dc3545',
            
            // Archived/Retired statuses - Gray
            'archived' => '#6c757d',
            'retired' => '#6c757d',
            'disposed' => '#6c757d',
            'scrapped' => '#6c757d',
            'obsolete' => '#6c757d',
            'decommissioned' => '#6c757d',
            
            // Information statuses - Blue
            'reserved' => '#17a2b8',
            'on hold' => '#17a2b8',
            'quarantined' => '#17a2b8',
            'testing' => '#17a2b8',
            'in testing' => '#17a2b8',
            
            // Special statuses - Purple
            'lost' => '#6f42c1',
            'stolen' => '#6f42c1',
            'missing' => '#6f42c1',
        ];
        
        // Check for exact match first
        if (isset($semanticColors[$statusLabel])) {
            return $semanticColors[$statusLabel];
        }
        
        // Check for partial matches
        foreach ($semanticColors as $key => $color) {
            if (strpos($statusLabel, $key) !== false || strpos($key, $statusLabel) !== false) {
                return $color;
            }
        }
        
        return null;
    }
}
