#!/bin/bash
# Snipe-IT Deployment Script for Namecheap Server
# Run this script on your server after uploading the files

echo "🚀 Starting Snipe-IT Deployment..."

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    print_error "Please run this script as root or with sudo"
    exit 1
fi

# Get domain name from user
read -p "Enter your domain name (e.g., snipeit.yourdomain.com): " DOMAIN_NAME
if [ -z "$DOMAIN_NAME" ]; then
    print_error "Domain name is required!"
    exit 1
fi

# Get database details
read -p "Enter database name: " DB_NAME
read -p "Enter database username: " DB_USER
read -s -p "Enter database password: " DB_PASS
echo
read -p "Enter database host (default: localhost): " DB_HOST
DB_HOST=${DB_HOST:-localhost}

# Get email settings
read -p "Enter SMTP host: " SMTP_HOST
read -p "Enter SMTP port (default: 587): " SMTP_PORT
SMTP_PORT=${SMTP_PORT:-587}
read -p "Enter SMTP username: " SMTP_USER
read -s -p "Enter SMTP password: " SMTP_PASS
echo

print_status "Setting up production environment..."

# Create production .env file
cat > .env << EOF
# Production Environment Configuration
APP_NAME="Snipe-IT"
APP_ENV=production
APP_DEBUG=false
APP_KEY=
APP_URL=https://$DOMAIN_NAME
APP_TIMEZONE='UTC'
APP_LOCALE='en-US'
MAX_RESULTS=500

# Database Settings
DB_CONNECTION=mysql
DB_HOST=$DB_HOST
DB_PORT=3306
DB_DATABASE=$DB_NAME
DB_USERNAME=$DB_USER
DB_PASSWORD=$DB_PASS
DB_PREFIX=null
DB_DUMP_PATH='/usr/bin'
DB_CHARSET=utf8mb4
DB_COLLATION=utf8mb4_unicode_ci

# Mail Settings
MAIL_MAILER=smtp
MAIL_HOST=$SMTP_HOST
MAIL_PORT=$SMTP_PORT
MAIL_USERNAME=$SMTP_USER
MAIL_PASSWORD=$SMTP_PASS
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDR=$SMTP_USER
MAIL_FROM_NAME='Snipe-IT System'
MAIL_REPLYTO_ADDR=$SMTP_USER
MAIL_REPLYTO_NAME='Snipe-IT Support'

# Image Library
IMAGE_LIB=gd

# Session Settings
SESSION_DRIVER=database
SESSION_LIFETIME=12000
EXPIRE_ON_CLOSE=false
ENCRYPT=false
COOKIE_NAME=snipeit_session
PASSPORT_COOKIE_NAME='snipeit_passport_token'
COOKIE_DOMAIN=.$DOMAIN_NAME
SECURE_COOKIES=true

# Security Settings
APP_TRUSTED_PROXIES=*
ALLOW_IFRAMING=false
REFERRER_POLICY=strict-origin-when-cross-origin
ENABLE_CSP=true
ENABLE_HSTS=true

# Cache Settings
CACHE_DRIVER=database
QUEUE_CONNECTION=database
CACHE_PREFIX=snipeit_prod

# Logging
LOG_CHANNEL=daily
LOG_MAX_DAYS=30
APP_LOCKED=false
APP_CIPHER=AES-256-CBC
APP_FORCE_TLS=true

# Performance Settings
LDAP_MEM_LIM=500M
LDAP_TIME_LIM=600
IMPORT_TIME_LIMIT=600
IMPORT_MEMORY_LIMIT=500M
REPORT_TIME_LIMIT=12000
API_THROTTLE_PER_MINUTE=60
CSV_ESCAPE_FORMULAS=true

# Hashing
HASHING_DRIVER='bcrypt'
BCRYPT_ROUNDS=12
EOF

print_status "Environment file created successfully!"

# Generate application key
print_status "Generating application key..."
php artisan key:generate --force

# Set proper permissions
print_status "Setting file permissions..."
chown -R www-data:www-data .
chmod -R 755 .
chmod -R 775 storage bootstrap/cache

# Clear and cache configuration
print_status "Optimizing application..."
php artisan config:cache
php artisan route:cache
php artisan view:cache

# Run database migrations
print_status "Running database migrations..."
php artisan migrate --force

# Create storage symlinks
print_status "Creating storage symlinks..."
php artisan storage:link

print_status "Deployment completed successfully!"
print_warning "Don't forget to:"
print_warning "1. Configure your web server (Apache/Nginx) to point to the 'public' directory"
print_warning "2. Set up SSL certificate for HTTPS"
print_warning "3. Configure firewall rules"
print_warning "4. Set up automated backups"
print_warning "5. Test the application at https://$DOMAIN_NAME"

echo
print_status "🎉 Snipe-IT is ready to use!"
