/**
 * Enhanced Navigation JavaScript for Mobility-IT
 * Adds interactive features and smooth animations to the sidebar navigation
 */

(function($) {
    'use strict';

    // Enhanced Navigation Class
    class EnhancedNavigation {
        constructor() {
            this.init();
        }

        init() {
            this.setupEventListeners();
            this.initializeTooltips();
            this.setupSmoothScrolling();
            this.initializeProgressBars();
            this.setupKeyboardNavigation();
            this.initializeSearchEnhancement();
        }

        setupEventListeners() {
            console.log('Enhanced Navigation: Setting up event listeners...');
            
            // Listen for existing push-menu events to understand the system
            $('body').on('expanded.pushMenu', function() {
                console.log('Enhanced Navigation: Push-menu expanded event fired');
            });
            
            $('body').on('collapsed.pushMenu', function() {
                console.log('Enhanced Navigation: Push-menu collapsed event fired');
            });
            
            // Enhanced hover effects for menu items
            $('.sidebar-menu > li > a').on('mouseenter', function() {
                $(this).addClass('hover-effect');
            }).on('mouseleave', function() {
                $(this).removeClass('hover-effect');
            });

            // Enhanced sidebar toggle button interactions
            $('.enhanced-sidebar-toggle').on('click', function(e) {
                console.log('Enhanced Navigation: Toggle button clicked!');
                console.log('Event object:', e);
                console.log('Current body classes:', $('body').attr('class'));
                
                // DON'T prevent default - let the existing push-menu system work
                const $this = $(this);
                const $body = $('body');
                
                // Add click animation
                $this.addClass('clicked');
                setTimeout(() => {
                    $this.removeClass('clicked');
                }, 300);
                
                // Let the existing push-menu system handle the toggle
                // We'll just add visual feedback and track the state
                
                // Wait a bit for the existing system to process, then update our button state
                setTimeout(() => {
                    if ($body.hasClass('sidebar-collapse')) {
                        console.log('Enhanced Navigation: Sidebar is now collapsed');
                        $this.removeClass('sidebar-open');
                    } else {
                        console.log('Enhanced Navigation: Sidebar is now expanded');
                        $this.addClass('sidebar-open');
                    }
                    
                    // Debug logging
                    console.log('Body classes after toggle:', $body.attr('class'));
                    console.log('Sidebar collapsed state:', $body.hasClass('sidebar-collapse'));
                    
                    // Check if the push-menu system is working
                    console.log('Push-menu events fired:', {
                        expanded: $body.data('push-menu-expanded'),
                        collapsed: $body.data('push-menu-collapsed')
                    });
                }, 100);
            });
            
            // Enhanced hover effects for toggle button
            $('.enhanced-sidebar-toggle').on('mouseenter', function() {
                $(this).addClass('hover-enhanced');
            }).on('mouseleave', function() {
                $(this).removeClass('hover-enhanced');
            });
            


            // Enhanced click effects
            $('.sidebar-menu a').on('click', function() {
                $(this).addClass('click-effect');
                setTimeout(() => {
                    $(this).removeClass('click-effect');
                }, 200);
            });

            // Enhanced treeview animations
            $('.treeview > a').on('click', function(e) {
                const $li = $(this).parent();
                const $submenu = $li.find('.treeview-menu');
                
                if ($submenu.length) {
                    e.preventDefault();
                    
                    if ($li.hasClass('active')) {
                        $li.removeClass('active');
                        $submenu.slideUp(300);
                    } else {
                        $('.treeview.active').removeClass('active').find('.treeview-menu').slideUp(300);
                        $li.addClass('active');
                        $submenu.slideDown(300);
                    }
                }
            });

            // Enhanced badge animations
            $('.sidebar-menu .badge').on('mouseenter', function() {
                $(this).addClass('badge-hover');
            }).on('mouseleave', function() {
                $(this).removeClass('badge-hover');
            });
        }

        initializeTooltips() {
            // Enhanced tooltip system
            $('.menu-tooltip').each(function() {
                const $this = $(this);
                const tooltipText = $this.attr('data-tooltip');
                
                if (tooltipText) {
                    $this.on('mouseenter', function(e) {
                        this.showTooltip(e, tooltipText);
                    }).on('mouseleave', function() {
                        this.hideTooltip();
                    });
                }
            });
        }

        showTooltip(event, text) {
            // Remove existing tooltips
            $('.enhanced-tooltip').remove();
            
            const $tooltip = $(`
                <div class="enhanced-tooltip">
                    <div class="tooltip-content">${text}</div>
                    <div class="tooltip-arrow"></div>
                </div>
            `);
            
            $('body').append($tooltip);
            
            const $target = $(event.currentTarget);
            const targetRect = $target[0].getBoundingClientRect();
            const tooltipRect = $tooltip[0].getBoundingClientRect();
            
            // Position tooltip
            $tooltip.css({
                position: 'fixed',
                left: targetRect.right + 10,
                top: targetRect.top + (targetRect.height / 2) - (tooltipRect.height / 2),
                zIndex: 10000
            });
            
            $tooltip.addClass('show');
        }

        hideTooltip() {
            $('.enhanced-tooltip').remove();
        }

        setupSmoothScrolling() {
            // Smooth scrolling for sidebar
            $('.main-sidebar').on('scroll', function() {
                const scrollTop = $(this).scrollTop();
                const $sidebar = $(this);
                
                if (scrollTop > 10) {
                    $sidebar.addClass('scrolled');
                } else {
                    $sidebar.removeClass('scrolled');
                }
            });
        }

        initializeProgressBars() {
            // Animate progress bars on view
            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const $progressBar = $(entry.target).find('.progress-bar');
                        if ($progressBar.length) {
                            const width = $progressBar.css('width');
                            $progressBar.css('width', '0%');
                            setTimeout(() => {
                                $progressBar.css('width', width);
                            }, 100);
                        }
                    }
                });
            });

            $('.menu-progress').each(function() {
                observer.observe(this);
            });
        }

        setupKeyboardNavigation() {
            // Enhanced keyboard navigation
            $('.sidebar-menu a').on('keydown', function(e) {
                const $current = $(this);
                const $parent = $current.parent();
                const $siblings = $parent.siblings();
                
                switch(e.keyCode) {
                    case 37: // Left arrow
                        e.preventDefault();
                        if ($parent.hasClass('treeview') && $parent.hasClass('active')) {
                            $parent.removeClass('active').find('.treeview-menu').slideUp(300);
                        }
                        break;
                    case 39: // Right arrow
                        e.preventDefault();
                        if ($parent.hasClass('treeview') && !$parent.hasClass('active')) {
                            $parent.addClass('active').find('.treeview-menu').slideDown(300);
                        }
                        break;
                    case 38: // Up arrow
                        e.preventDefault();
                        const $prev = $parent.prev();
                        if ($prev.length) {
                            $prev.find('a').focus();
                        }
                        break;
                    case 40: // Down arrow
                        e.preventDefault();
                        const $next = $parent.next();
                        if ($next.length) {
                            $next.find('a').focus();
                        }
                        break;
                    case 13: // Enter
                    case 32: // Space
                        e.preventDefault();
                        $current[0].click();
                        break;
                }
            });
        }

        initializeSearchEnhancement() {
            // Enhanced search functionality
            const $searchInput = $('.sidebar-search input');
            const $searchForm = $('.sidebar-search form');
            
            if ($searchInput.length) {
                // Auto-focus search on keyboard shortcut
                $(document).on('keydown', function(e) {
                    if ((e.ctrlKey || e.metaKey) && e.keyCode === 70) { // Ctrl/Cmd + F
                        e.preventDefault();
                        $searchInput.focus();
                    }
                });
                
                // Enhanced search with suggestions
                $searchInput.on('input', function() {
                    const query = $(this).val().toLowerCase();
                    this.filterMenuItems(query);
                });
                
                // Clear search on escape
                $searchInput.on('keydown', function(e) {
                    if (e.keyCode === 27) { // Escape
                        $(this).val('').trigger('input');
                        $(this).blur();
                    }
                });
            }
        }

        filterMenuItems(query) {
            if (query.length < 2) {
                $('.sidebar-menu li').show();
                return;
            }
            
            $('.sidebar-menu li').each(function() {
                const $li = $(this);
                const text = $li.text().toLowerCase();
                const hasMatch = text.includes(query);
                
                if (hasMatch) {
                    $li.show();
                    $li.addClass('search-highlight');
                } else {
                    $li.hide();
                }
            });
        }

        // Public methods for external use
        refresh() {
            this.init();
        }

        toggleSidebar() {
            $('body').toggleClass('sidebar-collapse');
        }

        expandAll() {
            $('.treeview').addClass('active').find('.treeview-menu').show();
        }

        collapseAll() {
            $('.treeview').removeClass('active').find('.treeview-menu').hide();
        }
    }

    // Initialize when document is ready
    $(document).ready(function() {
        console.log('Enhanced Navigation: Initializing...');
        
        // Check if enhanced toggle button exists
        if ($('.enhanced-sidebar-toggle').length) {
            console.log('Enhanced Navigation: Toggle button found');
        } else {
            console.log('Enhanced Navigation: Toggle button NOT found');
        }
        
        window.enhancedNavigation = new EnhancedNavigation();
        
        // Add some CSS classes for enhanced styling
        $('body').addClass('enhanced-navigation-loaded');
        
        // Add loading states
        $('.sidebar-menu > li').each(function(index) {
            $(this).css('animation-delay', (index * 0.1) + 's');
        });
        
        console.log('Enhanced Navigation: Initialization complete');
    });

    // Add CSS for enhanced features
    const enhancedStyles = `
        <style>
            .enhanced-tooltip {
                position: absolute;
                background: #2c3e50;
                color: #ffffff;
                padding: 8px 12px;
                border-radius: 6px;
                font-size: 12px;
                white-space: nowrap;
                z-index: 10000;
                opacity: 0;
                transform: translateX(-10px);
                transition: all 0.3s ease;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
            }
            
            .enhanced-tooltip.show {
                opacity: 1;
                transform: translateX(0);
            }
            
            .enhanced-tooltip .tooltip-arrow {
                position: absolute;
                left: -5px;
                top: 50%;
                transform: translateY(-50%);
                border: 5px solid transparent;
                border-right-color: #2c3e50;
            }
            
            .search-highlight {
                background: rgba(52, 152, 219, 0.1) !important;
                border-left: 2px solid #3498db !important;
            }
            
            .hover-effect {
                transform: translateX(8px) !important;
            }
            
            .click-effect {
                transform: scale(0.95) !important;
            }
            
            .badge-hover {
                transform: scale(1.2) !important;
                box-shadow: 0 6px 12px rgba(0, 0, 0, 0.4) !important;
            }
            
            .scrolled {
                box-shadow: 4px 0 20px rgba(0, 0, 0, 0.2) !important;
            }
            
            .enhanced-navigation-loaded .sidebar-menu > li {
                animation: slideInLeft 0.5s ease forwards;
            }
            
            @keyframes slideInLeft {
                from {
                    opacity: 0;
                    transform: translateX(-30px);
                }
                to {
                    opacity: 1;
                    transform: translateX(0);
                }
            }
        </style>
    `;
    
    $('head').append(enhancedStyles);

})(jQuery);
