/**
 * Heroicons Navigation Implementation for Mobility-IT
 * Modern, professional icon system
 */

class HeroiconsNavigation {
    constructor() {
        this.icons = {
            // Dashboard - Chart bars representing overview
            dashboard: `<svg class="nav-icon icon-dashboard" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
            </svg>`,
            
            // Assets - Computer representing hardware
            assets: `<svg class="nav-icon icon-assets" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9.75 17L9 20l-1 1h8l-1-1-.75-3M3 13h18M5 17h14a2 2 0 002-2V5a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"/>
            </svg>`,
            
            // Licenses - Document representing legal documents
            licenses: `<svg class="nav-icon icon-licenses" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"/>
            </svg>`,
            
            // Accessories - Headphones representing peripheral devices
            accessories: `<svg class="nav-icon icon-accessories" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z"/>
            </svg>`,
            
            // Consumables - Flask representing supplies
            consumables: `<svg class="nav-icon icon-consumables" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19.428 15.428a2 2 0 00-1.022-.547l-2.387-.477a6 6 0 00-3.86.517l-.318.158a6 6 0 01-3.86.517L6.05 15.21a2 2 0 00-1.806.547M8 4h8l-1 1v5.172a2 2 0 00.586 1.414l5 5c1.26 1.26.367 3.414-1.415 3.414H4.828c-1.782 0-2.674-2.154-1.414-3.414l5-5A2 2 0 009 10.172V5L8 4z"/>
            </svg>`,
            
            // Components - Cog representing technical parts
            components: `<svg class="nav-icon icon-components" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/>
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
            </svg>`,
            
            // Kits - Package representing bundled items
            kits: `<svg class="nav-icon icon-kits" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4"/>
            </svg>`,
            
            // Users - Multiple people representing user management
            users: `<svg class="nav-icon icon-users" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z"/>
            </svg>`,
            
            // Import - Arrow down representing data import
            import: `<svg class="nav-icon icon-import" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M9 19l3 3m0 0l3-3m-3 3V10"/>
            </svg>`,
            
            // Settings - Cog representing configuration
            settings: `<svg class="nav-icon icon-settings" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z"/>
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
            </svg>`,
            
            // Reports - Chart representing analytics
            reports: `<svg class="nav-icon icon-reports" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"/>
            </svg>`,
            
            // Search - Magnifying glass
            search: `<svg class="nav-icon icon-search" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
            </svg>`,
            
            // Plus - Add new item
            plus: `<svg class="nav-icon icon-plus" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"/>
            </svg>`,
            
            // Checkout - Shopping cart representing checkout
            checkout: `<svg class="nav-icon icon-checkout" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 3h2l.4 2M7 13h10l4-8H5.4m0 0L7 13m0 0l-2.5 5M7 13l2.5 5m6-5v6a2 2 0 01-2 2H9a2 2 0 01-2-2v-6m8 0V9a2 2 0 00-2-2H9a2 2 0 00-2 2v4.01"/>
            </svg>`,
            
            // Alerts - Bell representing notifications
            alerts: `<svg class="nav-icon icon-alerts" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 17h5l-5 5v-5zM9 7h6m0 10v-3m-3 3h.01M9 17h.01M9 14h.01M12 14h.01M15 11h.01M12 11h.01M9 11h.01M7 21h10a2 2 0 002-2V5a2 2 0 00-2-2H7a2 2 0 00-2 2v14a2 2 0 002 2z"/>
            </svg>`,
            
            // User Menu - User representing account
            user: `<svg class="nav-icon icon-user-menu" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"/>
            </svg>`,
            
            // Angle Left - Chevron for dropdowns
            angleLeft: `<svg class="nav-icon icon-angle-left" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
            </svg>`,
            
            // Circle - Status indicators
            circle: `<svg class="nav-icon icon-circle small" viewBox="0 0 24 24" fill="currentColor">
                <circle cx="12" cy="12" r="10"/>
            </svg>`,
            
            // Checkmark - Success/confirmation
            checkmark: `<svg class="nav-icon icon-checkmark" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
            </svg>`,
            
            // X - Close/deny
            x: `<svg class="nav-icon icon-x" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
            </svg>`,
            
            // Due - Calendar representing due dates
            due: `<svg class="nav-icon icon-due" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"/>
            </svg>`,
            
            // Requestable - Hand representing requests
            requestable: `<svg class="nav-icon icon-requestable" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 11.5V14m0-2.5v-6a1.5 1.5 0 113 0m-3 6a1.5 1.5 0 00-3 0v2a7.5 7.5 0 0015 0v-5a1.5 1.5 0 00-3 0m-6-3V11a1.5 1.5 0 013 0v3m0 0V11a1.5 1.5 0 013 0v3m0 0V11a1.5 1.5 0 013 0v3"/>
            </svg>`,
            
            // Warning - Exclamation triangle
            warning: `<svg class="nav-icon icon-warning" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.34 16.5c-.77.833.192 2.5 1.732 2.5z"/>
            </svg>`,
            
            // Heart - Love/favorite
            heart: `<svg class="nav-icon icon-heart" viewBox="0 0 24 24" fill="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z"/>
            </svg>`,
            
            // Admin Settings - Shield representing security
            adminSettings: `<svg class="nav-icon icon-admin-settings" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"/>
            </svg>`,
            
            // Nav Toggle - Menu bars
            navToggle: `<svg class="nav-icon icon-nav-toggle" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"/>
            </svg>`,
            
            // API Key - Key representing authentication
            apiKey: `<svg class="nav-icon icon-api-key" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 7a2 2 0 012 2m4 0a6 6 0 01-7 7m4-7a6 6 0 00-7-7m-7 7a6 6 0 007 7m-7-7a6 6 0 017-7m-7 7a6 6 0 017 7"/>
            </svg>`,
            
            // Password - Lock representing security
            password: `<svg class="nav-icon icon-password" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"/>
            </svg>`,
            
            // Logout - Arrow right representing exit
            logout: `<svg class="nav-icon icon-logout" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1a3 3 0 01-3 3H6a3 3 0 01-3-3V7a3 3 0 013-3h4a3 3 0 013 3v1"/>
            </svg>`
        };
        
        this.init();
    }
    
    init() {
        console.log('Heroicons Navigation: Initializing...');
        this.cleanupOverlappingIcons();
        this.replaceAllIcons();
        this.addEventListeners();
        console.log('Heroicons Navigation: Initialized successfully!');
    }
    
    cleanupOverlappingIcons() {
        // Remove any duplicate or overlapping icons
        const allIcons = document.querySelectorAll('.nav-icon, svg.nav-icon');
        allIcons.forEach(icon => {
            // Remove duplicate classes
            const classes = icon.className.split(' ').filter((cls, index, arr) => arr.indexOf(cls) === index);
            icon.className = classes.join(' ');
            
            // Ensure proper positioning
            if (icon.classList.contains('pull-right')) {
                icon.style.marginLeft = 'auto';
                icon.style.marginRight = '0';
            }
            if (icon.classList.contains('pull-left')) {
                icon.style.marginRight = 'auto';
                icon.style.marginLeft = '0';
            }
        });
    }
    
    replaceAllIcons() {
        // Replace all x-icon components with Heroicons
        this.replaceXIconComponents();
        
        // Replace FontAwesome icons
        this.replaceFontAwesomeIcons();
        
        // Replace other icon types
        this.replaceOtherIcons();
    }
    
    replaceXIconComponents() {
        // Find all x-icon components and replace them
        const xIcons = document.querySelectorAll('x-icon');
        xIcons.forEach(icon => {
            const type = icon.getAttribute('type');
            if (type && this.icons[type]) {
                icon.outerHTML = this.icons[type];
            }
        });
    }
    
    replaceFontAwesomeIcons() {
        // Replace FontAwesome icons with Heroicons
        const faIcons = document.querySelectorAll('.fa, .fas, .far, .fab');
        faIcons.forEach(icon => {
            const iconClass = icon.className;
            let replacementIcon = null;
            
            // Map FontAwesome classes to Heroicon types
            if (iconClass.includes('fa-dashboard') || iconClass.includes('fa-tachometer-alt')) {
                replacementIcon = this.icons.dashboard;
            } else if (iconClass.includes('fa-desktop') || iconClass.includes('fa-laptop')) {
                replacementIcon = this.icons.assets;
            } else if (iconClass.includes('fa-file-text') || iconClass.includes('fa-certificate')) {
                replacementIcon = this.icons.licenses;
            } else if (iconClass.includes('fa-headphones') || iconClass.includes('fa-keyboard')) {
                replacementIcon = this.icons.accessories;
            } else if (iconClass.includes('fa-flask') || iconClass.includes('fa-tint')) {
                replacementIcon = this.icons.consumables;
            } else if (iconClass.includes('fa-cogs') || iconClass.includes('fa-cog')) {
                replacementIcon = this.icons.components;
            } else if (iconClass.includes('fa-users') || iconClass.includes('fa-user')) {
                replacementIcon = this.icons.users;
            } else if (iconClass.includes('fa-cog') || iconClass.includes('fa-wrench')) {
                replacementIcon = this.icons.settings;
            } else if (iconClass.includes('fa-search')) {
                replacementIcon = this.icons.search;
            } else if (iconClass.includes('fa-plus')) {
                replacementIcon = this.icons.plus;
            } else if (iconClass.includes('fa-shopping-cart')) {
                replacementIcon = this.icons.checkout;
            } else if (iconClass.includes('fa-bell')) {
                replacementIcon = this.icons.alerts;
            } else if (iconClass.includes('fa-user-circle')) {
                replacementIcon = this.icons.user;
            } else if (iconClass.includes('fa-angle-left') || iconClass.includes('fa-chevron-left')) {
                replacementIcon = this.icons.angleLeft;
                // Add specific class for angle-left icons
                if (iconClass.includes('pull-right')) {
                    replacementIcon = replacementIcon.replace('icon-angle-left', 'icon-angle-left pull-right');
                }
            } else if (iconClass.includes('fa-circle')) {
                replacementIcon = this.icons.circle;
            } else if (iconClass.includes('fa-check')) {
                replacementIcon = this.icons.checkmark;
            } else if (iconClass.includes('fa-times')) {
                replacementIcon = this.icons.x;
            } else if (iconClass.includes('fa-calendar')) {
                replacementIcon = this.icons.due;
            } else if (iconClass.includes('fa-hand-paper')) {
                replacementIcon = this.icons.requestable;
            } else if (iconClass.includes('fa-exclamation-triangle')) {
                replacementIcon = this.icons.warning;
            } else if (iconClass.includes('fa-heart')) {
                replacementIcon = this.icons.heart;
            } else if (iconClass.includes('fa-shield')) {
                replacementIcon = this.icons.adminSettings;
            } else if (iconClass.includes('fa-bars')) {
                replacementIcon = this.icons.navToggle;
            } else if (iconClass.includes('fa-key')) {
                replacementIcon = this.icons.apiKey;
            } else if (iconClass.includes('fa-lock')) {
                replacementIcon = this.icons.password;
            } else if (iconClass.includes('fa-sign-out-alt')) {
                replacementIcon = this.icons.logout;
            }
            
            if (replacementIcon) {
                // Create new icon element
                const newIcon = document.createElement('div');
                newIcon.innerHTML = replacementIcon;
                const svg = newIcon.querySelector('svg');
                
                // Clean up class names and preserve essential ones
                const originalClasses = icon.className.split(' ').filter(cls => 
                    cls && !cls.includes('fas') && !cls.includes('fa-') && !cls.includes('pull-right') && !cls.includes('pull-left')
                );
                const svgClasses = svg.className.split(' ').filter(cls => cls && cls !== 'nav-icon');
                const combinedClasses = [...originalClasses, ...svgClasses, 'nav-icon'].join(' ');
                
                svg.setAttribute('class', combinedClasses);
                
                // Copy essential attributes only
                const essentialAttrs = ['aria-hidden', 'title', 'data-toggle', 'data-target'];
                essentialAttrs.forEach(attrName => {
                    if (icon.hasAttribute(attrName)) {
                        svg.setAttribute(attrName, icon.getAttribute(attrName));
                    }
                });
                
                // Replace the icon
                icon.outerHTML = svg.outerHTML;
                
                // Apply positioning after replacement
                setTimeout(() => {
                    const newSvg = document.querySelector(`svg[class*="${svg.className.split(' ')[0]}"]`);
                    if (newSvg) {
                        if (newSvg.classList.contains('pull-right')) {
                            newSvg.style.marginLeft = 'auto';
                            newSvg.style.marginRight = '0';
                        }
                        if (newSvg.classList.contains('pull-left')) {
                            newSvg.style.marginRight = 'auto';
                            newSvg.style.marginLeft = '0';
                        }
                    }
                }, 10);
            }
        });
    }
    
    replaceOtherIcons() {
        // Replace any remaining icon types
        const otherIcons = document.querySelectorAll('i[class*="icon"]');
        otherIcons.forEach(icon => {
            // Add specific replacements for other icon types
            const iconClass = icon.className;
            if (iconClass.includes('icon-')) {
                // Handle custom icon classes
                console.log('Found custom icon:', iconClass);
            }
        });
    }
    
    addEventListeners() {
        // Add hover effects and interactions
        document.addEventListener('mouseover', (e) => {
            if (e.target.closest('.nav-icon')) {
                e.target.closest('.nav-icon').classList.add('pulse');
            }
        });
        
        document.addEventListener('mouseout', (e) => {
            if (e.target.closest('.nav-icon')) {
                e.target.closest('.nav-icon').classList.remove('pulse');
            }
        });
        
        // Add click effects
        document.addEventListener('click', (e) => {
            if (e.target.closest('.nav-icon')) {
                const icon = e.target.closest('.nav-icon');
                icon.classList.add('clicked');
                setTimeout(() => {
                    icon.classList.remove('clicked');
                }, 200);
            }
        });
        
        // Check for sidebar menu overflow and add scrollbar classes
        this.checkSidebarOverflow();
        
        // Listen for window resize to recheck overflow
        window.addEventListener('resize', () => {
            this.checkSidebarOverflow();
        });
    }
    
    checkSidebarOverflow() {
        // Check sidebar menu overflow
        const sidebarMenu = document.querySelector('.sidebar-menu');
        if (sidebarMenu) {
            // Check if menu content exceeds container height
            const hasOverflow = sidebarMenu.scrollHeight > sidebarMenu.clientHeight;
            
            if (hasOverflow) {
                sidebarMenu.classList.add('has-overflow');
            } else {
                sidebarMenu.classList.remove('has-overflow');
            }
            
            // Add smooth scrolling behavior
            sidebarMenu.style.scrollBehavior = 'smooth';
        }
        
        // Check main sidebar overflow
        const mainSidebar = document.querySelector('.main-sidebar');
        if (mainSidebar) {
            // Check if sidebar content exceeds viewport height
            const hasOverflow = mainSidebar.scrollHeight > mainSidebar.clientHeight;
            
            if (hasOverflow) {
                mainSidebar.classList.add('has-overflow');
            } else {
                mainSidebar.classList.remove('has-overflow');
            }
            
            // Add smooth scrolling behavior
            mainSidebar.style.scrollBehavior = 'smooth';
        }
    }
    
    // Method to manually replace a specific icon
    replaceIcon(selector, iconType) {
        const element = document.querySelector(selector);
        if (element && this.icons[iconType]) {
            element.innerHTML = this.icons[iconType];
        }
    }
    
    // Method to add a new icon type
    addIconType(name, svgContent) {
        this.icons[name] = svgContent;
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    window.heroiconsNavigation = new HeroiconsNavigation();
});

// Export for use in other scripts
if (typeof module !== 'undefined' && module.exports) {
    module.exports = HeroiconsNavigation;
}
