/**
 * Modern Date Picker Enhancement
 * Provides enhanced functionality for date inputs while maintaining compatibility
 */

class ModernDatePicker {
    constructor() {
        this.init();
    }

    init() {
        console.log('📅 Modern Date Picker: Initializing...');
        this.enhanceDateInputs();
        this.addEventListeners();
        this.setupKeyboardNavigation();
        this.addAccessibilityFeatures();
        console.log('📅 Modern Date Picker: Initialization complete');
    }

    enhanceDateInputs() {
        // Find all date inputs
        const dateInputs = document.querySelectorAll('input[type="date"], input[type="datetime-local"], input[type="time"], input[type="month"], input[type="week"]');
        
        dateInputs.forEach(input => {
            // Add modern-date-picker class for enhanced styling
            if (!input.closest('.modern-date-picker')) {
                const wrapper = document.createElement('div');
                wrapper.className = 'modern-date-picker';
                input.parentNode.insertBefore(wrapper, input);
                wrapper.appendChild(input);
            }

            // Add enhanced attributes
            input.setAttribute('data-enhanced', 'true');
            
            // Add placeholder if not present
            if (!input.placeholder) {
                const type = input.type;
                switch (type) {
                    case 'date':
                        input.placeholder = 'Select date';
                        break;
                    case 'datetime-local':
                        input.placeholder = 'Select date and time';
                        break;
                    case 'time':
                        input.placeholder = 'Select time';
                        break;
                    case 'month':
                        input.placeholder = 'Select month';
                        break;
                    case 'week':
                        input.placeholder = 'Select week';
                        break;
                }
            }

            // Add validation attributes
            this.addValidationAttributes(input);
        });
    }

    addValidationAttributes(input) {
        // Add min/max attributes for better UX
        const type = input.type;
        const today = new Date();
        
        if (type === 'date' || type === 'datetime-local') {
            // Set reasonable min date (e.g., 100 years ago)
            const minDate = new Date(today.getFullYear() - 100, 0, 1);
            if (!input.min) {
                input.min = minDate.toISOString().split('T')[0];
            }
            
            // Set reasonable max date (e.g., 10 years from now)
            const maxDate = new Date(today.getFullYear() + 10, 11, 31);
            if (!input.max) {
                input.max = maxDate.toISOString().split('T')[0];
            }
        }
    }

    addEventListeners() {
        // Listen for theme changes to update date picker styling
        document.addEventListener('themeChanged', () => {
            this.updateDatePickerStyling();
        });

        // Listen for date input changes
        document.addEventListener('change', (e) => {
            if (e.target.matches('input[type="date"], input[type="datetime-local"], input[type="time"], input[type="month"], input[type="week"]')) {
                this.handleDateChange(e.target);
            }
        });

        // Listen for date input focus
        document.addEventListener('focus', (e) => {
            if (e.target.matches('input[type="date"], input[type="datetime-local"], input[type="time"], input[type="month"], input[type="week"]')) {
                this.handleDateFocus(e.target);
            }
        }, true);

        // Listen for date input blur
        document.addEventListener('blur', (e) => {
            if (e.target.matches('input[type="date"], input[type="datetime-local"], input[type="time"], input[type="month"], input[type="week"]')) {
                this.handleDateBlur(e.target);
            }
        }, true);
    }

    handleDateChange(input) {
        // Add visual feedback for date selection
        input.classList.add('date-selected');
        
        // Remove the class after animation
        setTimeout(() => {
            input.classList.remove('date-selected');
        }, 300);

        // Validate the selected date
        this.validateDate(input);

        // Dispatch custom event
        const event = new CustomEvent('dateSelected', {
            detail: {
                input: input,
                value: input.value,
                type: input.type
            }
        });
        document.dispatchEvent(event);
    }

    handleDateFocus(input) {
        // Add focus styling
        input.classList.add('date-focused');
        
        // Show helpful tooltip if needed
        this.showDateTooltip(input);
    }

    handleDateBlur(input) {
        // Remove focus styling
        input.classList.remove('date-focused');
        
        // Hide tooltip
        this.hideDateTooltip(input);
    }

    validateDate(input) {
        const value = input.value;
        if (!value) return;

        const type = input.type;
        let isValid = true;
        let message = '';

        try {
            if (type === 'date') {
                const date = new Date(value);
                const today = new Date();
                
                // Check if date is valid
                if (isNaN(date.getTime())) {
                    isValid = false;
                    message = 'Please enter a valid date';
                }
                // Check if date is in the future (for certain use cases)
                else if (date > today && input.hasAttribute('data-no-future')) {
                    isValid = false;
                    message = 'Date cannot be in the future';
                }
            }
            else if (type === 'time') {
                const timeRegex = /^([0-1]?[0-9]|2[0-3]):[0-5][0-9]$/;
                if (!timeRegex.test(value)) {
                    isValid = false;
                    message = 'Please enter a valid time (HH:MM)';
                }
            }
        } catch (error) {
            isValid = false;
            message = 'Invalid date format';
        }

        // Update validation state
        this.updateValidationState(input, isValid, message);
    }

    updateValidationState(input, isValid, message) {
        // Remove existing validation classes
        input.classList.remove('is-valid', 'is-invalid', 'is-warning');
        
        if (isValid) {
            input.classList.add('is-valid');
        } else {
            input.classList.add('is-invalid');
            this.showValidationMessage(input, message);
        }
    }

    showValidationMessage(input, message) {
        // Remove existing validation message
        const existingMessage = input.parentNode.querySelector('.date-validation-message');
        if (existingMessage) {
            existingMessage.remove();
        }

        // Create new validation message
        const validationMessage = document.createElement('div');
        validationMessage.className = 'date-validation-message';
        validationMessage.textContent = message;
        validationMessage.style.cssText = `
            color: var(--danger-color);
            font-size: 12px;
            margin-top: 4px;
            opacity: 0;
            transition: opacity 0.3s ease;
        `;

        // Insert after input
        input.parentNode.insertBefore(validationMessage, input.nextSibling);

        // Animate in
        setTimeout(() => {
            validationMessage.style.opacity = '1';
        }, 10);
    }

    showDateTooltip(input) {
        const type = input.type;
        let tooltipText = '';

        switch (type) {
            case 'date':
                tooltipText = 'Click to open date picker';
                break;
            case 'datetime-local':
                tooltipText = 'Click to open date and time picker';
                break;
            case 'time':
                tooltipText = 'Click to open time picker';
                break;
            case 'month':
                tooltipText = 'Click to open month picker';
                break;
            case 'week':
                tooltipText = 'Click to open week picker';
                break;
        }

        // Create tooltip
        const tooltip = document.createElement('div');
        tooltip.className = 'date-tooltip';
        tooltip.textContent = tooltipText;
        tooltip.style.cssText = `
            position: absolute;
            top: -30px;
            left: 50%;
            transform: translateX(-50%);
            background: var(--card-bg);
            color: var(--text-primary);
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
            white-space: nowrap;
            z-index: 1000;
            border: 1px solid var(--border-primary);
            box-shadow: var(--card-shadow);
            opacity: 0;
            transition: opacity 0.3s ease;
            pointer-events: none;
        `;

        // Add to input wrapper
        const wrapper = input.closest('.modern-date-picker');
        if (wrapper) {
            wrapper.style.position = 'relative';
            wrapper.appendChild(tooltip);

            // Animate in
            setTimeout(() => {
                tooltip.style.opacity = '1';
            }, 10);
        }
    }

    hideDateTooltip(input) {
        const tooltip = input.closest('.modern-date-picker')?.querySelector('.date-tooltip');
        if (tooltip) {
            tooltip.style.opacity = '0';
            setTimeout(() => {
                tooltip.remove();
            }, 300);
        }
    }

    setupKeyboardNavigation() {
        // Add keyboard shortcuts for date inputs
        document.addEventListener('keydown', (e) => {
            const activeElement = document.activeElement;
            
            if (activeElement.matches('input[type="date"], input[type="datetime-local"], input[type="time"], input[type="month"], input[type="week"]')) {
                switch (e.key) {
                    case 'Enter':
                        // Open date picker on Enter
                        activeElement.showPicker?.();
                        e.preventDefault();
                        break;
                    case 'Escape':
                        // Clear date on Escape
                        if (e.ctrlKey || e.metaKey) {
                            activeElement.value = '';
                            activeElement.dispatchEvent(new Event('change'));
                            e.preventDefault();
                        }
                        break;
                    case 'ArrowUp':
                        // Increment date/time
                        this.incrementDate(activeElement, 1);
                        e.preventDefault();
                        break;
                    case 'ArrowDown':
                        // Decrement date/time
                        this.incrementDate(activeElement, -1);
                        e.preventDefault();
                        break;
                }
            }
        });
    }

    incrementDate(input, direction) {
        const type = input.type;
        const currentValue = input.value;
        
        if (!currentValue) return;

        try {
            if (type === 'date') {
                const date = new Date(currentValue);
                date.setDate(date.getDate() + direction);
                input.value = date.toISOString().split('T')[0];
            }
            else if (type === 'time') {
                const [hours, minutes] = currentValue.split(':').map(Number);
                const date = new Date();
                date.setHours(hours, minutes + (direction * 15)); // 15-minute increments
                input.value = date.toTimeString().slice(0, 5);
            }
            else if (type === 'datetime-local') {
                const date = new Date(currentValue);
                date.setMinutes(date.getMinutes() + (direction * 15));
                input.value = date.toISOString().slice(0, 16);
            }

            // Trigger change event
            input.dispatchEvent(new Event('change'));
        } catch (error) {
            console.warn('Date increment failed:', error);
        }
    }

    addAccessibilityFeatures() {
        // Add ARIA labels and descriptions
        const dateInputs = document.querySelectorAll('input[type="date"], input[type="datetime-local"], input[type="time"], input[type="month"], input[type="week"]');
        
        dateInputs.forEach(input => {
            // Add ARIA label if not present
            if (!input.getAttribute('aria-label') && !input.getAttribute('aria-labelledby')) {
                const label = input.closest('.form-group')?.querySelector('label');
                if (label) {
                    input.setAttribute('aria-labelledby', label.id || label.textContent);
                } else {
                    input.setAttribute('aria-label', `Select ${input.type}`);
                }
            }

            // Add ARIA description
            input.setAttribute('aria-describedby', `date-help-${input.id || Math.random().toString(36).substr(2, 9)}`);
        });
    }

    updateDatePickerStyling() {
        // Update styling when theme changes
        const dateInputs = document.querySelectorAll('input[type="date"], input[type="datetime-local"], input[type="time"], input[type="month"], input[type="week"]');
        
        dateInputs.forEach(input => {
            // Force re-render of calendar picker indicator
            const indicator = input.style;
            indicator.setProperty('--calendar-color', 'var(--accent-primary)');
        });
    }

    // Public methods for external use
    static enhanceAll() {
        return new ModernDatePicker();
    }

    static enhance(input) {
        if (input.matches('input[type="date"], input[type="datetime-local"], input[type="time"], input[type="month"], input[type="week"]')) {
            const picker = new ModernDatePicker();
            picker.enhanceDateInputs();
            return picker;
        }
        return null;
    }
}

// Auto-initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    ModernDatePicker.enhanceAll();
});

// Re-initialize when new content is added (for dynamic content)
const observer = new MutationObserver((mutations) => {
    mutations.forEach((mutation) => {
        mutation.addedNodes.forEach((node) => {
            if (node.nodeType === Node.ELEMENT_NODE) {
                const dateInputs = node.querySelectorAll ? node.querySelectorAll('input[type="date"], input[type="datetime-local"], input[type="time"], input[type="month"], input[type="week"]') : [];
                if (dateInputs.length > 0) {
                    ModernDatePicker.enhanceAll();
                }
            }
        });
    });
});

observer.observe(document.body, {
    childList: true,
    subtree: true
});

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ModernDatePicker;
}
