/**
 * Theme Toggle System for Mobility-IT
 * Handles switching between dark and light modes
 */

class ThemeManager {
    constructor() {
        this.currentTheme = this.getStoredTheme() || 'light';
        this.init();
    }

    /**
     * Initialize the theme system
     */
    init() {
        this.applyTheme(this.currentTheme);
        this.createThemeToggle();
        this.addEventListeners();
        console.log('Theme Manager initialized with theme:', this.currentTheme);
    }

    /**
     * Get stored theme from localStorage
     */
    getStoredTheme() {
        return localStorage.getItem('mobilityit-theme');
    }

    /**
     * Store theme preference in localStorage
     */
    storeTheme(theme) {
        localStorage.setItem('mobilityit-theme', theme);
    }

    /**
     * Apply theme to the document
     */
    applyTheme(theme) {
        document.documentElement.setAttribute('data-theme', theme);
        this.currentTheme = theme;
        this.storeTheme(theme);
        
        // Update theme toggle button if it exists
        const toggleBtn = document.querySelector('.theme-toggle');
        if (toggleBtn) {
            toggleBtn.setAttribute('data-theme', theme);
        }
        
        // Update navbar theme toggle
        this.updateNavbarThemeToggle();
        
        // Dispatch custom event for other components
        document.dispatchEvent(new CustomEvent('themeChanged', { 
            detail: { theme: theme } 
        }));
    }

    /**
     * Toggle between light and dark themes
     */
    toggleTheme() {
        const newTheme = this.currentTheme === 'light' ? 'dark' : 'light';
        this.applyTheme(newTheme);
        
        // Add animation effect
        this.addThemeTransitionEffect();
        
        console.log('Theme toggled to:', newTheme);
    }

    /**
     * Add smooth transition effect when changing themes
     */
    addThemeTransitionEffect() {
        document.body.style.transition = 'all 0.3s ease';
        setTimeout(() => {
            document.body.style.transition = '';
        }, 300);
    }

    /**
     * Create the theme toggle button
     */
    createThemeToggle() {
        // Check if toggle already exists
        if (document.querySelector('.theme-toggle')) {
            return;
        }

        const toggleBtn = document.createElement('button');
        toggleBtn.className = 'theme-toggle';
        toggleBtn.setAttribute('data-theme', this.currentTheme);
        toggleBtn.setAttribute('aria-label', 'Toggle theme');
        toggleBtn.setAttribute('title', 'Switch between light and dark mode');
        
        toggleBtn.innerHTML = `
            <i class="theme-icon fas ${this.currentTheme === 'light' ? 'fa-moon' : 'fa-sun'}"></i>
            <span class="theme-text">${this.currentTheme === 'light' ? 'Dark' : 'Light'} Mode</span>
        `;

        // Try to add to navbar first, fallback to body if navbar not found
        const navbar = document.querySelector('.navbar-nav');
        if (navbar) {
            // Create a list item for the navbar
            const listItem = document.createElement('li');
            listItem.className = 'theme-toggle-item';
            listItem.appendChild(toggleBtn);
            
            // Ensure it's positioned at the far right
            listItem.style.marginLeft = 'auto';
            listItem.style.order = '999';
            
            navbar.appendChild(listItem);
            
            // Add navbar class for styling
            toggleBtn.classList.add('navbar-theme-toggle');
        } else {
            // Fallback: add to the page body
            document.body.appendChild(toggleBtn);
        }
        
        // Position the button properly
        this.positionThemeToggle();
    }

    /**
     * Position the theme toggle button
     */
    positionThemeToggle() {
        const toggleBtn = document.querySelector('.theme-toggle');
        if (!toggleBtn) return;

        // If it's a navbar toggle, no positioning needed
        if (toggleBtn.classList.contains('navbar-theme-toggle')) {
            return;
        }

        // Adjust position based on sidebar state (only for fixed positioned buttons)
        const sidebar = document.querySelector('.main-sidebar');
        if (sidebar) {
            const isCollapsed = sidebar.classList.contains('sidebar-collapse');
            if (isCollapsed) {
                toggleBtn.style.right = '20px';
            } else {
                toggleBtn.style.right = '250px'; // Account for sidebar width
            }
        }

        // Adjust position on window resize
        window.addEventListener('resize', () => {
            this.positionThemeToggle();
        });
    }

    /**
     * Add event listeners
     */
    addEventListeners() {
        // Listen for sidebar collapse/expand events
        document.addEventListener('expanded.pushMenu', () => {
            this.positionThemeToggle();
        });

        document.addEventListener('collapsed.pushMenu', () => {
            this.positionThemeToggle();
        });

        // Listen for theme toggle clicks
        document.addEventListener('click', (e) => {
            if (e.target.closest('.theme-toggle')) {
                this.toggleTheme();
            }
        });

        // Listen for keyboard shortcuts
        document.addEventListener('keydown', (e) => {
            // Ctrl/Cmd + T to toggle theme
            if ((e.ctrlKey || e.metaKey) && e.key === 't') {
                e.preventDefault();
                this.toggleTheme();
            }
        });

        // Listen for navbar changes (mobile menu toggle)
        const navbarToggle = document.querySelector('.navbar-toggle');
        if (navbarToggle) {
            navbarToggle.addEventListener('click', () => {
                // Small delay to allow navbar to expand/collapse
                setTimeout(() => {
                    this.updateNavbarThemeToggle();
                }, 100);
            });
        }
    }

    /**
     * Get current theme
     */
    getCurrentTheme() {
        return this.currentTheme;
    }

    /**
     * Check if current theme is dark
     */
    isDarkTheme() {
        return this.currentTheme === 'dark';
    }

    /**
     * Check if current theme is light
     */
    isLightTheme() {
        return this.currentTheme === 'light';
    }

    /**
     * Force apply a specific theme
     */
    setTheme(theme) {
        if (['light', 'dark'].includes(theme)) {
            this.applyTheme(theme);
        }
    }

    /**
     * Reset theme to system preference
     */
    resetToSystemPreference() {
        const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
        const systemTheme = prefersDark ? 'dark' : 'light';
        this.applyTheme(systemTheme);
    }

    /**
     * Update navbar theme toggle button
     */
    updateNavbarThemeToggle() {
        const toggleBtn = document.querySelector('.navbar-theme-toggle');
        if (toggleBtn) {
            // Update button text and icon based on current theme
            const themeText = toggleBtn.querySelector('.theme-text');
            const themeIcon = toggleBtn.querySelector('.theme-icon');
            
            if (themeText) {
                themeText.textContent = this.currentTheme === 'light' ? 'Dark' : 'Light' + ' Mode';
            }
            
            if (themeIcon) {
                // Update icon to Font Awesome icons
                themeIcon.className = `theme-icon fas ${this.currentTheme === 'light' ? 'fa-moon' : 'fa-sun'}`;
            }
        }
    }

    /**
     * Listen for system theme changes
     */
    listenForSystemThemeChanges() {
        const mediaQuery = window.matchMedia('(prefers-color-scheme: dark)');
        mediaQuery.addEventListener('change', (e) => {
            if (!this.getStoredTheme()) { // Only auto-switch if user hasn't set preference
                this.applyTheme(e.matches ? 'dark' : 'light');
            }
        });
    }
}

// Auto-detect system theme preference
function detectSystemTheme() {
    const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
    return prefersDark ? 'dark' : 'light';
}

// Initialize theme manager when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    // Create global theme manager instance
    window.themeManager = new ThemeManager();
    
    // Auto-detect system theme if no preference is stored
    if (!localStorage.getItem('mobilityit-theme')) {
        const systemTheme = detectSystemTheme();
        window.themeManager.setTheme(systemTheme);
    }
    
    // Listen for system theme changes
    window.themeManager.listenForSystemThemeChanges();
});

// Export for module usage
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ThemeManager;
}
